#ifndef __IThread__
#define __IThread__

#include "../Basics/CCountedObject.hpp"
#include "IThreadListener.hpp"
#include "CCriticalSection.hpp"
using Exponent::Threading::CCriticalSection;
using Exponent::Threading::IThreadListener;
using Exponent::Basics::CCountedObject;

#ifndef WIN32
#include <pthread.h>
#endif

//	===========================================================================

namespace Exponent
{
	namespace Threading
	{
		/**
		 * @interface IThread IThread.hpp
		 * @brief A wrapper around a cross platform thread
		 *
		 * @date 13/01/2006
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: IThread.hpp,v 1.6 2007/02/08 21:06:44 paul Exp $
		 */
		interface IThread
		{
		public:

//	===========================================================================

			/**
			 * @enum EThreadPriority
			 * @brief Priority of the thread
			 */
			enum EThreadPriority
			{
				e_highPriority = 0,			/**< High priority, eg audio */
				e_normalPriority,			/**< Normal priority */
				e_lowPriority,				/**< Low priority thread */
			};

//	===========================================================================

			/**
			 * @struct SThreadHandle IThread.hpp
			 * @brief Cross platform thread handle
			 */
			struct SThreadHandle
			{
				#ifdef WIN32
					HANDLE m_threadHandle;				/**< The thread handle */
				#else
					pthread_t *m_threadHandle;			/**< The thread handle */
				#endif
			};
			
//	===========================================================================
			
			/**
		     * Construction
			 */
			IThread() { }
			
			/**
			 * Destruction
			 */
			virtual ~IThread() { }
			
//	===========================================================================
			
			/**
			 * Do the actual process. Normally in a threaded object you would overload this function and do some interesting stuff inside
			 * @retval bool True if you completed properly, false otherwise
			 */
			virtual bool runThread() = 0;

			/**
			 * Stop the thread
			 * @note This is a forced termination, upon reciept of this message you must terminate as quickly as possible
			 */
			virtual void stopThread() = 0;

			/**
			 * Is thread active
			 * @retval bool True if thread is running, false otherwise
			 */
			virtual bool isThreadActive() const = 0;

//	===========================================================================
			  
			/**
			 * Register the thread listener
			 * @param listener The thread listener
			 */
			virtual void registerThreadListener(IThreadListener *listener) = 0;

			/**
			 * Get the thread listener
			 * @retval IThreadListener* The thread listener
			 */
			virtual IThreadListener *getThreadListener() = 0;

//	===========================================================================

			/**
			 * Set the thread priority
			 * @param priority The thread priority
			 */
			virtual void setThreadPriority(const IThread::EThreadPriority priority) = 0;

			/**
			 * Get the thread priority
			 * @return IThread::EThreadPriority The thread priority
			 */
			virtual IThread::EThreadPriority getThreadPriority() const = 0;

//	===========================================================================

			/**
			 * Sleep the thread
			 * @param timeInMilliseconds The sleep time in msec
			 */
			virtual void sleepThread(const long timeInMilliseconds) = 0;

//	===========================================================================

			/**
			 * Set the Thread handle
			 * @param threadHandle The handle to the thread You now own this pointer
			 */
			virtual void setThreadHandle(IThread::SThreadHandle *threadHandle) = 0;

			/**
			 * Get the thread handle
			 * @retval IThread::SThreadHandle * Handle to the thread
			 */
			virtual IThread::SThreadHandle *getThreadHandle() = 0;

//	===========================================================================

			/**
			 * Set the critical section
			 * @param criticalSection The ciritical setion
			 */
			virtual void setCriticalSection(CCriticalSection *criticalSection) = 0;
		};
	}
}
#endif	// End of IThread.hpp